/** @file   menumain.cpp
 * @brief   Implementation of MenuMain - class.
 * @version $Revision: 1.4 $
 * @author  Tomi Lamminsaari
 */

#include "menumain.h"
#include "GfxManager.h"
#include "soundsamples.h"
#include "eng2d.h"
#include "helpnote.h"
#include "consts.h"
#include <string>
#include "datafile_mainmenu.h"

using std::string;
using namespace eng2d;
using std::vector;


namespace WeWantWar {

const int MenuMain::PLAY_THE_GAME;
const int MenuMain::GOTO_LOADGAME_MENU;
const int MenuMain::GOTO_HIGHSCORES_MENU;
const int MenuMain::GOTO_SETTINGS_MENU;
const int MenuMain::EXIT_THE_GAME;


/** Constructs the mainmenu-object
 */
MenuMain::MenuMain() :
  MenuPage()
{
  this->createHelpNotes();
  this->initGraphics();
}



/** Destructor
 */
MenuMain::~MenuMain()
{
  this->cleanupGraphics();
}


/** Updates the mainmenu. Checks the keyboard and makes the selections
 */
int MenuMain::update()
{
  int ret = MenuPage::NO_PAGE_CHANGE;
  if ( this->getState() == MenuPage::STATE_FADEIN ) {
    ret = this->updateFadeInState();
    
  } else if ( this->getState() == MenuPage::STATE_VISIBLE ) {
    ret = this->updateVisibleState();
    if ( ret == MenuPage::PAGE_CHANGE ) {
      MenuPage::m_state = STATE_FADEOUT;
    }
    ret = MenuPage::NO_PAGE_CHANGE;
    
  } else if ( this->getState() == MenuPage::STATE_FADEOUT ) {
    ret = this->updateFadeOutState();
    
  }
  return ret;
}



/** Redraws the mainmenu
 */
void MenuMain::redraw()
{
  // Draw background
  blit( GfxManager::titlePic, Display::buffer, 0,0, 0,0, 640,480 );
  
  // We print the version info
  textout_right_ex( Display::buffer, font, Consts::VERSION_STR.c_str(),
                    620,470, makecol(200,200,0),-1 );
  
  // Draw the menu alternatives
  
  for ( int i=0; i < 5; i++ ) {
    int index = i;
    
    if ( MenuPage::m_selection == i ) {
      index = i + 5;
    }
    Sprite* pS = m_sprites.at( index );
    pS->draw( Display::buffer );
  }
  
  // Draw little images
  RLE_SPRITE* pS = m_sprites.at( SP_GRENADE + MenuPage::m_selection)->al_RLESPRITE();
  Vec2D p( m_sprites.at(SP_GRENADE + MenuPage::m_selection)->position() );
  int x = static_cast<int>( p.x() - ( pS->w / 2 ) );
  int y = static_cast<int>( p.y() - ( pS->h / 2 ) );
  draw_rle_sprite( Display::buffer, pS, x, y );
}


/** Updates the fade in state
 */
int MenuMain::updateFadeInState()
{
  // Scroll the menuitems from right
  Vec2D pos = m_sprites.at(0)->position();
  float nx = pos.x() - 20;
  pos.x( nx );
  for ( int i=0; i < 5; i++ ) {
    pos = m_sprites.at(i)->position();
    pos.x( nx );
    m_sprites.at(i)->position( pos );
    m_sprites.at(i+5)->position( pos );
  }
  if ( nx < 450 ) {
    MenuPage::m_state = MenuPage::STATE_VISIBLE;
  }
  
  // Fix the little offset problem with the selected EXIT GAME - option
  Vec2D fixPos( m_sprites.at(SP_EXIT_SEL)->position() );
  fixPos.vy -= 1;
  m_sprites.at(SP_EXIT_SEL)->position( fixPos );
  return MenuPage::NO_PAGE_CHANGE;
}



/** Updates the normal state
 */
int MenuMain::updateVisibleState()
{
  // Based on which menuitem has the selection we either fade the help notes
  // in or out.
  int index = 0;
  while ( true ) {
    OverlayItem* pItem = this->getOverlayItem( index );
    if ( pItem == 0 ) {
      break;
    }

    if ( index == this->getSelection() ) {
      pItem->fadeIn( 8 );

    } else {
      pItem->fadeOut( 4 );

    }
    index += 1;
  }
  
  int ret = MenuPage::NO_PAGE_CHANGE;
  
  // Check keyboard
  if ( key[MenuPage::s_lastKey] ) {
    return ret;
  }
  MenuPage::s_lastKey = KEY_O;


  if ( key[KEY_UP] ) {
    MenuPage::s_lastKey = KEY_UP;
    MenuPage::m_selection -= 1;
    if ( MenuPage::m_selection < 0 ) {
      MenuPage::m_selection = 4;
    }
    Sound::playSample( SMP_MENUITEM, false );

  } else if ( key[KEY_DOWN] ) {
    MenuPage::s_lastKey = KEY_DOWN;
    MenuPage::m_selection += 1;
    if ( MenuPage::m_selection > 4 ) {
      MenuPage::m_selection = 0;
    }
    Sound::playSample( SMP_MENUITEM, false );

  } else if ( key[KEY_SPACE] ) {
    MenuPage::s_lastKey = KEY_SPACE;
    ret = MenuPage::PAGE_CHANGE;

  } else if ( key[KEY_ENTER] ) {
    MenuPage::s_lastKey = KEY_ENTER;
    ret = MenuPage::PAGE_CHANGE;

  }

  if ( ret == MenuPage::PAGE_CHANGE ) {
    Sound::playSample(SMP_MENUSELECT, false);
  }
  return ret;
}



/** Updates the fadeout state
 */
int MenuMain::updateFadeOutState()
{
  Vec2D pos = m_sprites.at(0)->position();
  float nx = pos.x() + 20;
  for ( int i=0; i < 5; i++ ) {
    pos = m_sprites.at(i)->position();
    pos.x( nx );
    m_sprites.at(i)->position( pos );
  }
  if ( nx > 640 ) {
    return MenuPage::PAGE_CHANGE;
  }
  return MenuPage::NO_PAGE_CHANGE;
}



/** Creates the help notes
 */
void MenuMain::createHelpNotes()
{
  std::vector<string>* pTT = new std::vector<string>();

  pTT->push_back( "Starts the game. Get ready to" );
  pTT->push_back( "be eaten" );
  HelpNote* pItem = new HelpNote( 460, 420, 160, 40, *pTT );
  pItem->setOpacityRange( 0, 180 );
  this->addOverlayItem( pItem );
  delete pTT;

  pTT = new std::vector<string>();
  pTT->push_back( "Continue your previusly saved" );
  pTT->push_back( "game" );
  pItem = new HelpNote( 460, 420, 160, 40, *pTT );
  pItem->setOpacityRange( 0, 180 );
  this->addOverlayItem( pItem );
  delete pTT;
  
  pTT = new std::vector<string>();
  pTT->push_back( "View highscore tables." );
  pItem = new HelpNote( 460,420, 160,40, *pTT );
  pItem->setOpacityRange( 0, 180 );
  this->addOverlayItem( pItem );
  delete pTT;

  pTT = new std::vector<string>();
  pTT->push_back( "Change the game settings" );
  pItem = new HelpNote( 460, 420, 160, 40, *pTT );
  pItem->setOpacityRange( 0, 180 );
  this->addOverlayItem( pItem );
  delete pTT;

  pTT = new std::vector<string>();
  pTT->push_back( "Exit to Windows" );
  pItem = new HelpNote( 460, 420, 160, 40, *pTT );
  pItem->setOpacityRange( 0, 180 );
  this->addOverlayItem( pItem );
  
  delete pTT;
}



/** Initializes the graphics
 */
void MenuMain::initGraphics()
{
  m_pDat = load_datafile( "gfx/menus/mainmenu.dat" );

  // Create the sprites
  m_sprites.push_back( new Sprite( (RLE_SPRITE*)m_pDat[START].dat ) );
  m_sprites.push_back( new Sprite( (RLE_SPRITE*)m_pDat[LOAD].dat ) );
  m_sprites.push_back( new Sprite( (RLE_SPRITE*)m_pDat[HIGHSCORES].dat ) );
  m_sprites.push_back( new Sprite( (RLE_SPRITE*)m_pDat[SETTINGS].dat ) );
  m_sprites.push_back( new Sprite( (RLE_SPRITE*)m_pDat[EXIT].dat ) );
  
  m_sprites.push_back( new Sprite( (RLE_SPRITE*)m_pDat[START_SEL].dat ) );
  m_sprites.push_back( new Sprite( (RLE_SPRITE*)m_pDat[LOAD_SEL].dat ) );
  m_sprites.push_back( new Sprite( (RLE_SPRITE*)m_pDat[HIGHSCORE_SEL].dat ) );
  m_sprites.push_back( new Sprite( (RLE_SPRITE*)m_pDat[SETTINGS_SEL].dat ) );
  m_sprites.push_back( new Sprite( (RLE_SPRITE*)m_pDat[EXIT_SEL].dat ) );
  
  m_sprites.push_back( new Sprite( (RLE_SPRITE*)m_pDat[GRENADE].dat ) );
  m_sprites.push_back( new Sprite( (RLE_SPRITE*)m_pDat[HARDDISK].dat ) );
  m_sprites.push_back( new Sprite( (RLE_SPRITE*)m_pDat[TROPHY].dat ) );
  m_sprites.push_back( new Sprite( (RLE_SPRITE*)m_pDat[MONKEYWRENCH].dat ) );
  m_sprites.push_back( new Sprite( (RLE_SPRITE*)m_pDat[GRAVESTONE].dat ) );
  
  // Position the menuitem sprites
  int x = 640;
  int y = 112;
  for ( int i=0; i < 5; i++ ) {
    m_sprites.at(i)->position( Vec2D( x,y ) );
    m_sprites.at(i + SP_STARTGAME_SEL)->position( Vec2D( x,y ) );
    y += 40;
  }
  Vec2D pos( m_sprites.at(SP_EXIT_SEL)->position() );
  float ny = pos.y() - 3;
  pos.y( ny );
  m_sprites.at(SP_EXIT_SEL)->position( pos );
  
  // Position the little image sprites
  x = 200;
  y = 340;
  for ( int i=SP_GRENADE; i < SP_GRENADE + 5; i++ ) {
    m_sprites.at(i)->position( Vec2D( x,y ) );
  }
}



/** Removes the graphics
 */
void MenuMain::cleanupGraphics()
{
  unload_datafile( m_pDat );
}



} // end of namespace
